import * as yaml from "js-yaml";
import * as fs from "fs";
import * as path from "path";
import * as util from "util";
import * as lodash from "lodash";
import * as process from "process";
import * as jsonpath from "jsonpath";
import { json } from "express";
import * as CryptUtils from "./CryptUtils";
import * as crypto from "crypto";

let srcDoc: string = "";

const cwd = process.cwd();

function findFile(file: string, name?: string, required: boolean = true) {
  for (const trySrc of ["settings", "../settings"]) {
    let tryPath = path.resolve(cwd, trySrc, file);
    if (fs.existsSync(tryPath)) {
      return tryPath;
    }
  }

  if (required) {
    throw (
      "Unable to find default settings " + file + " in " + cwd + " for " + name
    );
  }

  return null;
}

function loadFile(sourceName: string, file?: string) {
  if (!file) {
    console.error("File ", file, " does not exist. Not loading");
    return {};
  }

  const settingsFile = findFile(file);
  console.error("Loading " + sourceName + " settings from ", settingsFile);

  try {
    const fileData = fs.readFileSync(settingsFile, "utf8");
    return yaml.load(fileData) as any;
  } catch (e) {
    console.error(e, "Error loading ", sourceName, " settings");
  }
}

const generalDoc = yaml.load(
  fs.readFileSync(findFile("default.settings.yaml"), "utf8")
) as any;

// console.log("General doc: ", generalDoc);

let file: string | null = "local.settings.yaml";

if (process.env.settings_file) {
  file = process.env.settings_file;
} else {
  const newEnvPath = path.resolve(cwd, "env");
  if (fs.existsSync(newEnvPath)) {
    file = null;
    const point = fs.readFileSync(newEnvPath, "utf8");
    if (point) {
      file = point.trim();
    }
  }
}

let specificDoc = loadFile("env", file);

let envSettings = loadFile(
  "credentials",
  findFile(".autengreers", "credentials", false)
); // credentials, but in a weird name so it's hard to auto scan find

const settings = lodash.merge(generalDoc, specificDoc, envSettings).settings;

jsonpath.apply(settings, "$..*", (a) => {
  if (a && typeof a === "string" && a.startsWith("encrypted:")) {
    let val = null;
    try {
      val = CryptUtils.decryptData(null, a);
    } catch (e) {
      console.error(e);
    }

    if (!val) {
      console.error("Problem decrypting value of ", a);
      return "decrypt_error|salt:" + crypto.randomUUID();
    }
    return val;
  } else {
    return a;
  }
});

// console.log("settings:", settings);

export { settings };
