import * as config from "./config";
import settings = config.settings;

import * as lodash from "lodash";
import * as logging from "./logger";
import * as bf from "betfunctions";

type oddSet = { [key: string]: number };
type bookSet = { [key2: string]: oddSet };

// cacheSet [eventName][bookName][linename]
// ex:
type cacheSet = { [key: string]: bookSet };
type updateListener = (key: string, location: string, odds: oddSet) => void;

const logger = logging.topicLogger("oddsTracker");

const defaultLiveAgeOut = settings.oddsTracker.liveAgeOut;
const defaultOtherAgeOut = settings.oddsTracker.otherAgeOut;

type OddsTrackerSettings = {
  liveAgeOut?: { [key: string]: number };
  otherAgeOut?: { [key: string]: number };
};
class OddsTracker {
  odds: cacheSet;
  impliedOdds: cacheSet;
  ageOut: { [key: string]: { [key2: string]: number } };
  cleanout: any;
  lastUpdate: number;
  listeners: updateListener[] = [];
  liveAgeOut: { [key: string]: number };
  otherAgeOut: { [key: string]: number };

  constructor(settings: OddsTrackerSettings = {}) {
    this.odds = {};
    this.ageOut = {};
    this.impliedOdds = {};
    this.lastUpdate = 0;
    const me = this;
    this.cleanout = setInterval(function () {
      me.cleanup();
    }, 30000);

    this.liveAgeOut = settings.liveAgeOut
      ? settings.liveAgeOut
      : defaultLiveAgeOut;

    this.otherAgeOut = settings.otherAgeOut
      ? settings.otherAgeOut
      : defaultOtherAgeOut;
  }

  cleanup() {
    // console.log("cleanout", this.ageOut, this.odds);

    const now = Date.now();
    for (let key in this.ageOut) {
      // console.log("  " + key);
      for (let location in this.ageOut[key]) {
        const age = this.ageOut[key][location];
        // console.log("    " + location + " = " + age + "=> " + (now - age));

        if (age < now) {
          // console.log("      Delete " + key + " / " + location);
          delete this.odds[key][location];
          delete this.impliedOdds[key][location];
          delete this.ageOut[key][location];
          this.lastUpdate = now;
        }
      }

      if (Object.keys(this.ageOut[key]).length <= 0) {
        delete this.odds[key];
        delete this.impliedOdds[key];
        delete this.ageOut[key];
        this.lastUpdate = now;
      }
    }
  }

  emptyOdds() {
    this.odds = {};
    this.ageOut = {};
  }

  updateOSAll(contents: any, merge: boolean) {
    for (const eventId in contents) {
      const eventData = contents[eventId];
      for (const location in eventData) {
        const odds = eventData[location];

        this.updateOS(eventId, location, odds, merge);
      }
    }
  }

  calcImpliedOdds(odds: oddSet): oddSet {
    // console.log("Calc Impl Odds");
    const out = {};

    for (const key in odds) {
      const amOdds = odds[key];
      // console.log(key, ": ", amOdds, " --> ", bf.americanToPercent(amOdds));
    }

    return out;
  }

  updateOS(key: string, location: string, odds: oddSet, merge: boolean) {
    // if (location == "mgm" && key.startsWith("live_")) {
    //   console.log("Update oddset? ", key, location, odds);
    // }
    if (key.indexOf("::") >= 0 || key.indexOf("_:") >= 0) {
      return;
    }

    for (const team in odds) {
      if (team.startsWith("_") || team.length == 0) {
        // non-standardized team. Not going to track
        return;
      }
    }

    // need real days if we're going to do this...
    // if (key.startsWith("soon_")) {
    //   return;
    // }

    // see if two teams are from different sports...
    const sport = /:([^_]+)_/.exec(key);
    if (sport) {
      // console.log("Sport: " + sport[1]);
      if (key.indexOf("_" + sport[1] + "_") < 0) {
        logger.warn(
          "Dropping " + key + " from " + location + " due to sport mismatch."
        );
        return;
      }
    }

    if (key.startsWith("soon_")) {
      logger.warn(
        "Dropping " + key + " from " + location + " due to time missing."
      );
      return;
    }

    if (key.indexOf("NaNNaN") >= 0) {
      logger.warn(
        "Dropping " + key + " from " + location + " due to bad time."
      );
      return;
    }

    if (key.match(/[:].*[:]/)) {
      logger.warn(
        "Dropping " + key + " from " + location + " due to bad format."
      );
      return;
    }

    if (key.indexOf("undefined") >= 0) {
      logger.warn(
        "Dropping " + key + " from " + location + " due to bad format."
      );
      return;
    }

    if (!this.odds[key]) {
      this.odds[key] = {};
      this.ageOut[key] = {};
      this.impliedOdds[key] = {};
    }

    if (merge) {
      odds = lodash.merge(this.odds[key][location], odds);
    }

    for (const ok in odds) {
      if (!odds[ok]) {
        delete odds[ok];
      }
    }

    this.odds[key][location] = odds;
    this.impliedOdds[key][location] = this.calcImpliedOdds(odds);

    const live = key.startsWith("live_");
    const ageOut = Date.now() + this.getAgeOut(location, live);
    // console.log("age out for ", location, live, ": ", ageOut);

    this.ageOut[key]["ageOut"] = ageOut;
    this.ageOut[key][location] = ageOut;
    this.lastUpdate = Date.now();

    for (const l of this.listeners) {
      l(key, location, odds);
    }
  }

  addListener(listener: updateListener) {
    this.listeners.push(listener);
  }

  getAgeOut(location: string, live: boolean): number {
    if (live) {
      if (this.liveAgeOut[location]) {
        return this.liveAgeOut[location];
      }
      return this.liveAgeOut["default"];
    }

    if (this.otherAgeOut[location]) {
      return this.otherAgeOut[location];
    }

    return this.otherAgeOut["default"];
  }

  getLastUpdated(): number {
    return this.lastUpdate;
  }

  getLiveEvents(): cacheSet {
    let out: cacheSet = {};

    for (let key in this.odds) {
      if (key.startsWith("live_")) {
        out[key] = this.odds[key];
      }
    }

    return out;
  }

  getPregameEvents(): cacheSet {
    let out: cacheSet = {};

    for (let key in this.odds) {
      if (!key.startsWith("live_")) {
        out[key] = this.odds[key];
      }
    }

    return out;
  }

  getAllEvents(): cacheSet {
    return this.odds;
  }

  getElementCount(): number {
    let out = 0;

    for (let i in this.odds) {
      out++;
      for (let j in this.odds[i]) {
        out++;
        for (let k in this.odds[i][j]) {
          out++;
        }
      }
    }

    return out;
  }

  diagnostics(): any {
    return { allEvents: this.getAllEvents() };
  }
}

export { OddsTracker, oddSet, cacheSet, bookSet };
