import * as OddsTracker from "./OddsTracker";

interface OddsSink {
  setMoneyline(standardizedTeam: string, odds: number): void;
  setSpread(standardizedTeam: string, spread: number, odds: number): void;
  setOver(total: number, odds: number): void;
  setUnder(total: number, odds: number): void;
  /**
   * Convienence method for over/under set. Negative means under, positive over.
   * @param total The strike. Positive for over, Negative for under
   * @param odds american odds
   */
  setTotal(total: number, odds: number): void;
}

class MoneylineOddsCollectionSink implements OddsSink {
  odds: OddsTracker.oddSet = {};

  setMoneyline(standardizedTeam: string, odds: number): void {
    this.odds[standardizedTeam] = odds;
  }

  setSpread(standardizedTeam: string, spread: number, odds: number): void {
    if (!isNaN(spread)) {
      let spreadStr = ("" + spread) as string;
      if (spread > 0 && spreadStr.indexOf("+") < 0) {
        spreadStr = "+" + spreadStr;
      }

      this.odds["$" + standardizedTeam + spreadStr] = odds;
    }
  }

  setOver(total: number, odds: number): void {
    if (!isNaN(total)) {
      this.odds["$over" + total] = odds;
    }
  }

  setUnder(total: number, odds: number): void {
    if (!isNaN(total)) {
      this.odds["$under" + total] = odds;
    }
  }

  setTotal(total: number, odds: number): void {
    if (total < 0) {
      this.setUnder(Math.abs(total), odds);
    } else {
      this.setOver(total, odds);
    }
  }

  getOdds() {
    return this.odds;
  }
}

export { MoneylineOddsCollectionSink, OddsSink };
