import * as WebSocket from "ws";
import * as express from "express";
import * as http from "http";

import * as logging from "./logger";

type Authenticator = (message: any, ws: WebSocket) => string;

class MessagingSocketServer {
  wss: WebSocket.Server;
  messageHandlers: Map<string, Function> = new Map();
  connectionListeners: Array<Function> = [];
  authenticationListeners: Array<Function> = [];
  logger: any;
  authenticator: Authenticator = null;
  cleanupinterval: NodeJS.Timeout;

  constructor(logname: string) {
    this.logger = logging.topicLogger(logname);

    const me = this;
    this.cleanupinterval = setInterval(function ping() {
      if (me.wss) {
        me.wss.clients.forEach(function each(ws) {
          // @ts-ignore
          if (!ws.isAlive) {
            me.logger.info(
              "Ending session for ",
              // @ts-ignore
              ws.clientid,
              new Date()
            );
            return ws.terminate();
          }

          // @ts-ignore
          ws.isAlive = false;
          ws.ping();
        });
      }
    }, 30000);
  }

  setAuthenticator(authenticator: Authenticator) {
    this.authenticator = authenticator;
  }

  addConnectionListener(listener: Function) {
    this.connectionListeners.push(listener);
  }

  addAuthenticationListener(listener: Function) {
    this.authenticationListeners.push(listener);
  }

  registerMessageHandler(type: string, handler) {
    this.messageHandlers.set(type, handler);
  }

  broadcast(type: string, message: any) {
    const strMsg = JSON.stringify({ type, message });
    // console.log("Sending " + strMsg);
    this.wss.clients.forEach((client) => {
      if (
        client.readyState === WebSocket.OPEN &&
        (this.authenticator == null ||
          // @ts-ignore
          client.clientid)
      ) {
        client.send(strMsg);
      }
    });
  }

  forEach(f: (ws: WebSocket) => void) {
    this.wss.clients.forEach(f);
  }

  open(path: string, server: http.Server) {
    this.wss = new WebSocket.WebSocketServer({
      path: path,
      server: server,
    });

    this.wss.on("connection", (ws) => {
      // @ts-ignore
      ws.isAlive = true;
      // @ts-ignore
      ws.clientid = null;

      ws.on("pong", () => {
        // @ts-ignore
        ws.isAlive = true;
      });

      ws.on("error", (e) => {
        this.logger.warn("WebSocket error:", e);
      });

      ws.on("message", (message) => {
        let parsedMessage = { type: "ERROR", message: {} };
        try {
          parsedMessage = JSON.parse(message.toString());
        } catch (e) {
          this.logger.warn("Problem parsing message", e);
        }

        if (
          parsedMessage.type === "login" ||
          parsedMessage.type == "startclient"
        ) {
          // @ts-ignore
          ws.clientid = authenticator(parsedMessage.message, ws);
          this.authenticationListeners.forEach((listener) => {
            listener(ws); // @ts-ignore
          });
        }

        // @ts-ignore
        if (this.authenticator && !ws.clientid) {
          this.logger.warn(
            "Attempted request without login ",
            JSON.stringify(message)
          );
          ws.close();
        } else {
          const handler = this.messageHandlers.get(parsedMessage.type);
          handler(parsedMessage.message);
        }
      });

      this.connectionListeners.forEach((listener) => {
        listener(ws);
      });
    });
  }
}

export { MessagingSocketServer };
