import * as crypto from "crypto";

const encryption_method = "aes256";

function orDefault(val: string, def: string) {
  return val ? val : def;
}

const markerText = "encrypted:";

const secret_keys = {};
for (const key in process.env) {
  const prefix = "CONFIG_SECRET_";
  if (key.startsWith(prefix)) {
    const keyPart = key.slice(prefix.length).toLowerCase();
    secret_keys[keyPart] = process.env[key];
  }
}

secret_keys["default"] = orDefault(
  secret_keys["default"],
  "DEV_PASSWORD_JjFvfdZNF8ar"
);

// orDefault(process.env.CONFIG_SECRET, "DEV_PASSWORD");

if (!secret_keys["default"] || !encryption_method) {
  throw new Error("secretKey and encryption_method are required");
}

function getKeyId(val) {
  if (!val) {
    return "default";
  }

  if (val.match(/^[:][a-z_]+$/)) {
    const out = val.slice(1);

    return out;
  }

  return null;
}
// Generate secret hash with crypto to use for encryption

function getKey(val: string) {
  const keyLookup = getKeyId(val);

  if (keyLookup) {
    // console.log("key lookup ", val, "  -->  ", keyLookup);
    val = secret_keys[keyLookup];
    // console.log("newval ", val);
  }

  if (val == null) {
    val = ":totalrubbish";
  }

  const key = crypto
    .createHash("sha512")
    .update(val)
    .digest("hex")
    .substring(0, 32);

  return key;
}

function getIV(val: string) {
  const encryptionIV = crypto
    .createHash("sha512")
    .update(val)
    .digest("hex")
    .substring(0, 16);

  return encryptionIV;
}

function encode64(val: string) {
  return Buffer.from(val).toString("base64");
}

function decode64(val: string) {
  return Buffer.from(val, "base64").toString("utf8");
}

export function addTestKey(name: string, val: string) {
  secret_keys[name] = val;
}

// Encrypt data
export function encryptData(secretKey: string, ivWord: string, data: string) {
  if (!ivWord || ivWord.length < 16 || ivWord.indexOf(",") >= 0) {
    throw new Error(
      "IV not specified correctly. Must be non null, >= 16 characters, cannot contain a ,"
    );
  }

  const cipher = crypto.createCipheriv(
    encryption_method,
    getKey(secretKey),
    getIV(ivWord)
  );

  const encData =
    cipher.update(data, "utf8", "base64") + cipher.final("base64");
  const object: any = { iv: ivWord, data: encData };
  const keyid = getKeyId(secretKey);

  // console.log(secretKey, "  -->  ", keyid);

  if (keyid && keyid != "default") {
    object.keyid = keyid;
  }

  // console.log(object);

  return markerText + encode64(JSON.stringify(object));
}

// Decrypt data
export function decryptData(secretKey: string, encryptedData: string) {
  encryptedData = encryptedData.trim();
  if (encryptedData.startsWith(markerText)) {
    encryptedData = encryptedData.slice(markerText.length);
  }

  const ob = JSON.parse(decode64(encryptedData));

  const ivWord = ob.iv;
  const buff = ob.data;
  const keyid = ob.keyid;

  if (!secretKey) {
    if (keyid) {
      secretKey = ":" + keyid;
    } else {
      secretKey = ":default";
    }
  }

  // console.log(secretKey, " --> ");
  const plainKey = getKey(secretKey);

  // console.log(secretKey, " -- ", plainKey);

  const decipher = crypto.createDecipheriv(
    encryption_method,
    plainKey,
    getIV(ivWord)
  );

  try {
    return (
      decipher.update(buff.toString("utf8"), "base64", "utf8") +
      decipher.final("utf8")
    ); // Decrypts data and converts to utf8
  } catch (e) {
    return null;
  }
}
