"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OddsTracker = void 0;
const config = require("./config");
var settings = config.settings;
const lodash = require("lodash");
const logging = require("./logger");
const logger = logging.topicLogger("oddsTracker");
const defaultLiveAgeOut = settings.oddsTracker.liveAgeOut;
const defaultOtherAgeOut = settings.oddsTracker.otherAgeOut;
class OddsTracker {
    constructor(settings = {}) {
        this.listeners = [];
        this.odds = {};
        this.ageOut = {};
        this.impliedOdds = {};
        this.lastUpdate = 0;
        const me = this;
        this.cleanout = setInterval(function () {
            me.cleanup();
        }, 30000);
        this.liveAgeOut = settings.liveAgeOut
            ? settings.liveAgeOut
            : defaultLiveAgeOut;
        this.otherAgeOut = settings.otherAgeOut
            ? settings.otherAgeOut
            : defaultOtherAgeOut;
    }
    cleanup() {
        // console.log("cleanout", this.ageOut, this.odds);
        const now = Date.now();
        for (let key in this.ageOut) {
            // console.log("  " + key);
            for (let location in this.ageOut[key]) {
                const age = this.ageOut[key][location];
                // console.log("    " + location + " = " + age + "=> " + (now - age));
                if (age < now) {
                    // console.log("      Delete " + key + " / " + location);
                    delete this.odds[key][location];
                    delete this.impliedOdds[key][location];
                    delete this.ageOut[key][location];
                    this.lastUpdate = now;
                }
            }
            if (Object.keys(this.ageOut[key]).length <= 0) {
                delete this.odds[key];
                delete this.impliedOdds[key];
                delete this.ageOut[key];
                this.lastUpdate = now;
            }
        }
    }
    emptyOdds() {
        this.odds = {};
        this.ageOut = {};
    }
    updateOSAll(contents, merge) {
        for (const eventId in contents) {
            const eventData = contents[eventId];
            for (const location in eventData) {
                const odds = eventData[location];
                this.updateOS(eventId, location, odds, merge);
            }
        }
    }
    calcImpliedOdds(odds) {
        // console.log("Calc Impl Odds");
        const out = {};
        for (const key in odds) {
            const amOdds = odds[key];
            // console.log(key, ": ", amOdds, " --> ", bf.americanToPercent(amOdds));
        }
        return out;
    }
    updateOS(key, location, odds, merge) {
        // if (location == "mgm" && key.startsWith("live_")) {
        //   console.log("Update oddset? ", key, location, odds);
        // }
        if (key.indexOf("::") >= 0 || key.indexOf("_:") >= 0) {
            return;
        }
        for (const team in odds) {
            if (team.startsWith("_") || team.length == 0) {
                // non-standardized team. Not going to track
                return;
            }
        }
        // need real days if we're going to do this...
        // if (key.startsWith("soon_")) {
        //   return;
        // }
        // see if two teams are from different sports...
        const sport = /:([^_]+)_/.exec(key);
        if (sport) {
            // console.log("Sport: " + sport[1]);
            if (key.indexOf("_" + sport[1] + "_") < 0) {
                logger.warn("Dropping " + key + " from " + location + " due to sport mismatch.");
                return;
            }
        }
        if (key.startsWith("soon_")) {
            logger.warn("Dropping " + key + " from " + location + " due to time missing.");
            return;
        }
        if (key.indexOf("NaNNaN") >= 0) {
            logger.warn("Dropping " + key + " from " + location + " due to bad time.");
            return;
        }
        if (key.match(/[:].*[:]/)) {
            logger.warn("Dropping " + key + " from " + location + " due to bad format.");
            return;
        }
        if (key.indexOf("undefined") >= 0) {
            logger.warn("Dropping " + key + " from " + location + " due to bad format.");
            return;
        }
        if (!this.odds[key]) {
            this.odds[key] = {};
            this.ageOut[key] = {};
            this.impliedOdds[key] = {};
        }
        if (merge) {
            odds = lodash.merge(this.odds[key][location], odds);
        }
        for (const ok in odds) {
            if (!odds[ok]) {
                delete odds[ok];
            }
        }
        this.odds[key][location] = odds;
        this.impliedOdds[key][location] = this.calcImpliedOdds(odds);
        const live = key.startsWith("live_");
        const ageOut = Date.now() + this.getAgeOut(location, live);
        // console.log("age out for ", location, live, ": ", ageOut);
        this.ageOut[key]["ageOut"] = ageOut;
        this.ageOut[key][location] = ageOut;
        this.lastUpdate = Date.now();
        for (const l of this.listeners) {
            l(key, location, odds);
        }
    }
    addListener(listener) {
        this.listeners.push(listener);
    }
    getAgeOut(location, live) {
        if (live) {
            if (this.liveAgeOut[location]) {
                return this.liveAgeOut[location];
            }
            return this.liveAgeOut["default"];
        }
        if (this.otherAgeOut[location]) {
            return this.otherAgeOut[location];
        }
        return this.otherAgeOut["default"];
    }
    getLastUpdated() {
        return this.lastUpdate;
    }
    getLiveEvents() {
        let out = {};
        for (let key in this.odds) {
            if (key.startsWith("live_")) {
                out[key] = this.odds[key];
            }
        }
        return out;
    }
    getPregameEvents() {
        let out = {};
        for (let key in this.odds) {
            if (!key.startsWith("live_")) {
                out[key] = this.odds[key];
            }
        }
        return out;
    }
    getAllEvents() {
        return this.odds;
    }
    getElementCount() {
        let out = 0;
        for (let i in this.odds) {
            out++;
            for (let j in this.odds[i]) {
                out++;
                for (let k in this.odds[i][j]) {
                    out++;
                }
            }
        }
        return out;
    }
    diagnostics() {
        return { allEvents: this.getAllEvents() };
    }
}
exports.OddsTracker = OddsTracker;
