"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MessagingSocketServer = void 0;
const WebSocket = require("ws");
const logging = require("./logger");
class MessagingSocketServer {
    constructor(logname) {
        this.messageHandlers = new Map();
        this.connectionListeners = [];
        this.authenticationListeners = [];
        this.authenticator = null;
        this.logger = logging.topicLogger(logname);
        const me = this;
        this.cleanupinterval = setInterval(function ping() {
            if (me.wss) {
                me.wss.clients.forEach(function each(ws) {
                    // @ts-ignore
                    if (!ws.isAlive) {
                        me.logger.info("Ending session for ", 
                        // @ts-ignore
                        ws.clientid, new Date());
                        return ws.terminate();
                    }
                    // @ts-ignore
                    ws.isAlive = false;
                    ws.ping();
                });
            }
        }, 30000);
    }
    setAuthenticator(authenticator) {
        this.authenticator = authenticator;
    }
    addConnectionListener(listener) {
        this.connectionListeners.push(listener);
    }
    addAuthenticationListener(listener) {
        this.authenticationListeners.push(listener);
    }
    registerMessageHandler(type, handler) {
        this.messageHandlers.set(type, handler);
    }
    broadcast(type, message) {
        const strMsg = JSON.stringify({ type, message });
        // console.log("Sending " + strMsg);
        this.wss.clients.forEach((client) => {
            if (client.readyState === WebSocket.OPEN &&
                (this.authenticator == null ||
                    // @ts-ignore
                    client.clientid)) {
                client.send(strMsg);
            }
        });
    }
    forEach(f) {
        this.wss.clients.forEach(f);
    }
    open(path, server) {
        this.wss = new WebSocket.WebSocketServer({
            path: path,
            server: server,
        });
        this.wss.on("connection", (ws) => {
            // @ts-ignore
            ws.isAlive = true;
            // @ts-ignore
            ws.clientid = null;
            ws.on("pong", () => {
                // @ts-ignore
                ws.isAlive = true;
            });
            ws.on("error", (e) => {
                this.logger.warn("WebSocket error:", e);
            });
            ws.on("message", (message) => {
                let parsedMessage = { type: "ERROR", message: {} };
                try {
                    parsedMessage = JSON.parse(message.toString());
                }
                catch (e) {
                    this.logger.warn("Problem parsing message", e);
                }
                if (parsedMessage.type === "login" ||
                    parsedMessage.type == "startclient") {
                    // @ts-ignore
                    ws.clientid = authenticator(parsedMessage.message, ws);
                    this.authenticationListeners.forEach((listener) => {
                        listener(ws); // @ts-ignore
                    });
                }
                // @ts-ignore
                if (this.authenticator && !ws.clientid) {
                    this.logger.warn("Attempted request without login ", JSON.stringify(message));
                    ws.close();
                }
                else {
                    const handler = this.messageHandlers.get(parsedMessage.type);
                    handler(parsedMessage.message);
                }
            });
            this.connectionListeners.forEach((listener) => {
                listener(ws);
            });
        });
    }
}
exports.MessagingSocketServer = MessagingSocketServer;
